<?php

namespace App\Http\Controllers;

use App\Package;
use Illuminate\Http\Request;
use Razorpay\Api\Api;
use App\Movie;
use App\Season; // Changed from TvSeries to Season
use App\PpvPurchase;
use App\Gst;
use App\Config;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class PayViaRazorpayController extends Controller
{
    public function success(Request $request, $planid, $type)
    {
        Log::info('Razorpay callback received', [
            'request_data' => $request->all(),
            'planid' => $planid,
            'type' => $type
        ]);
        
        try {
            // Initialize Razorpay API
            $api = new Api(env('RAZOR_PAY_KEY'), env('RAZOR_PAY_SECRET'));
            $gst = Gst::first();
            
            // Fetch payment information from Razorpay using the payment ID
            $payment = $api->payment->fetch($request->razorpay_payment_id);
            Log::info('Razorpay payment fetched', ['payment' => json_encode($payment)]);
            
            $config = Config::findOrFail(1);
            $base_currency = $config->currency_code;

            // Check if the payment exists and is successful
            if ($payment) {
                $payment_id = $payment->id;
                $payment_amount = $payment->amount / 100; // Razorpay returns amount in paise, convert to base unit
                $payment_method = 'razorpay';
                $payment_status = 1; // Assuming the payment is successful
                
                Log::info('Payment details', [
                    'payment_id' => $payment_id,
                    'payment_amount' => $payment_amount,
                    'payment_currency' => $payment->currency ?? 'N/A',
                    'payment_status' => $payment->status ?? 'N/A'
                ]);

                // Check the payment type
                if ($type == 'subscription') {
                    Log::info('Processing subscription payment', ['plan_id' => $planid]);
                    // Handle Subscription Logic
                    $plan = Package::findOrFail($planid);
                    $checkout = new SubscriptionController;
                    return $checkout->subscribe(
                        $payment_id,
                        $payment_method,
                        $plan->id,
                        $payment_status,
                        $payment_amount
                    );
                } elseif ($type == 'rental') {
                    // Handle PPV Logic (for Movies)
                    $movie = Movie::findOrFail($planid);
                    
                    $converted_price = (Session::has('current_currency'))
                        ? round(currency($movie->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($movie->price, 2);

                    $converted_offer_price = null;
                    
                    if (isset($movie->offer_price) && $movie->offer_price != null) {
                        $converted_offer_price = (Session::has('current_currency'))
                            ? round(currency($movie->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                            : round($movie->offer_price, 2);
                    }
                    
                    // Calculate GST and total amount
                    $base_price = $movie->offer_price > 0 ? $converted_offer_price : $converted_price;

                    if ($gst->gst_enable) {
                        // Calculate GST amount and total with GST
                        $gst_amount = ($base_price * $gst->gst_per) / 100;
                        $total_with_gst = $base_price + $gst_amount;
                    } else {
                        // If GST is disabled, set amount to base price without GST
                        $gst_amount = 0;
                        $total_with_gst = $base_price;
                    }
                    
                    Log::info('Movie rental payment details', [
                        'movie_id' => $movie->id,
                        'movie_title' => $movie->title,
                        'base_price' => $base_price,
                        'gst_percentage' => $gst->gst_enable,
                        'gst_amount' => $gst_amount,
                        'total_with_gst' => $total_with_gst,
                        'payment_amount' => $payment_amount,
                        'converted_price' => $converted_price,
                        'converted_offer_price' => $converted_offer_price,
                        'current_currency' => Session::get('current_currency') ?? $base_currency
                    ]);
                    
                    // Default expiration time for PPV purchase
                    $hours = $movie->hours;
                    $expiresAt = now()->addHours($hours);

                    // Check if payment amount matches the total amount including GST
                    if (abs($payment_amount - $total_with_gst) > 0.01) {
                        Log::error('Payment amount mismatch for movie rental', [
                            'expected' => $total_with_gst,
                            'received' => $payment_amount,
                            'difference' => $total_with_gst - $payment_amount
                        ]);
                        
                        // Despite the mismatch, we'll still process the payment if it's close enough
                        if (abs($payment_amount - $total_with_gst) > 1.00) {
                            return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                        }
                    }

                    // Save PPV purchase in the database
                    $purchase = PpvPurchase::create([
                        'user_id' => auth()->id(),
                        'movie_id' => $movie->id,
                        'payment_id' => $payment_id,
                        'gst' => $gst->gst_enable ? $gst->gst_per : 0,
                        'price' => $converted_price,
                        'offer_price' => $converted_offer_price ?? $converted_price,
                        'hours' => $hours,
                        'expires_at' => $expiresAt,
                        'type' => 'movie',
                        'payment_type' => 'RazorPay',
                        'status' => 1, // Active status
                    ]);
                    
                    Log::info('PPV purchase created', ['purchase_id' => $purchase->id]);

                    if (getSubscription()->getData()->subscribed == true) {
                        // Subscription is active
                        $route = 'movie/detail/';
                    } else {
                        // Subscription is not active
                        $route = 'movie/guest/detail/';
                    }               
                    return redirect()->to($route . $movie->slug)->with('success', __('Movie purchase successful!'));
                } elseif ($type == 'rental_series') {
                    // Handle PPV Logic (for Seasons)
                    $series = Season::findOrFail($planid); // Changed from TvSeries to Season
                    
                    $converted_price = (Session::has('current_currency'))
                        ? round(currency($series->price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($series->price, 2);
                    $converted_offer_price = null;
                    if (isset($series->offer_price) && $series->offer_price != null) {
                        $converted_offer_price = (Session::has('current_currency'))
                            ? round(currency($series->offer_price, $from = $base_currency, $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                            : round($series->offer_price, 2);
                    }
                    
                    // Calculate GST and total amount
                    $base_price = $series->offer_price > 0 ? $converted_offer_price : $converted_price;
                    
                    if ($gst->gst_enable) {
                        $gst_amount = ($base_price * $gst->gst_per) / 100;
                        $total_with_gst = $base_price + $gst_amount;
                    } else {
                        $gst_amount = 0;
                        $total_with_gst = $base_price;
                    }
                    
                    Log::info('Season rental payment details', [
                        'season_id' => $series->id,
                        'season_title' => 'Season ' . $series->season_no, // Adjusted for season
                        'base_price' => $base_price,
                        'gst_percentage' => $gst->gst_per,
                        'gst_amount' => $gst_amount,
                        'total_with_gst' => $total_with_gst,
                        'payment_amount' => $payment_amount,
                        'converted_price' => $converted_price,
                        'converted_offer_price' => $converted_offer_price,
                        'current_currency' => Session::get('current_currency') ?? $base_currency
                    ]);
                    
                    // Default expiration time for PPV purchase
                    $hours = $series->hours; // Assuming Season model has hours field
                    $expiresAt = now()->addHours($hours);
                    
                    // Check if payment amount matches the total amount including GST
                    if (abs($payment_amount - $total_with_gst) > 0.01) {
                        Log::error('Payment amount mismatch for season rental', [
                            'expected' => $total_with_gst,
                            'received' => $payment_amount,
                            'difference' => $total_with_gst - $payment_amount
                        ]);
                        
                        // Despite the mismatch, we'll still process the payment if it's close enough
                        if (abs($payment_amount - $total_with_gst) > 1.00) {
                            return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                        }
                    }

                    // Save PPV purchase in the database
                    $purchase = PpvPurchase::create([
                        'user_id' => auth()->id(),
                        'tv_series_id' => $series->id, // Changed from tv_series_id to season_id
                        'payment_id' => $payment_id,
                        'gst' => $gst->gst_per,
                        'price' => $converted_price,
                        'offer_price' => $converted_offer_price ?? $converted_price,
                        'hours' => $hours,
                        'expires_at' => $expiresAt,
                        'type' => 'series', // Kept as 'series' to maintain existing variable
                        'payment_type' => 'RazorPay',
                        'status' => 1, // Active status
                    ]);
                    
                    Log::info('PPV purchase created for season', ['purchase_id' => $purchase->id]);

                    if (getSubscription()->getData()->subscribed == true) {
                        // Subscription is active
                        $route = 'show/detail/';
                    } else {
                        // Subscription is not active
                        $route = 'show/guest/detail/';
                    }
                    return redirect()->to($route . $series->season_slug)->with('success', __('Season rental successful!')); // Adjusted to use season_slug
                } else {
                    // If the type is unknown, return error
                    Log::error('Invalid payment type', ['type' => $type]);
                    return redirect('/')->with('error', __('Invalid payment type.'));
                }
            } else {
                // Payment failed, return error
                Log::error('Payment not found or failed', ['razorpay_payment_id' => $request->razorpay_payment_id]);
                return redirect('/')->with('error', __('Payment failed.'));
            }
        } catch (\Exception $e) {
            Log::error('Exception in Razorpay payment process', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return redirect('/')->with('error', __('Payment processing error: ') . $e->getMessage());
        }
    }
}